<?php
/**
 * Plugin Name: wgsconnector
 * Plugin URL: https://wgsconnector.com/
 * Description: Connect WooCommerce to Google Sheets Automatically!
 * Version: 1.1.0
 * Update URI: https://api.freemius.com
 * Requires at least: 5.2
 * Requires PHP: 7.2
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Author: The wgsconnector Team 
 * Author URI: https://wgsconnector.com/
 */
if ( ! function_exists( 'wgs_fs' ) ) {
    // Create a helper function for easy SDK access.
    function wgs_fs() {
        global $wgs_fs;

        if ( ! isset( $wgs_fs ) ) {
            // Include Freemius SDK.
            require_once dirname( __FILE__ ) . '/vendor/freemius/start.php';
            $wgs_fs = fs_dynamic_init( array(
                'id'                  => '18902',
                'slug'                => 'wgsconnector',
                'type'                => 'plugin',
                'public_key'          => 'pk_2de243babcd539647e129a87cf582',
                'is_premium'          => true,
                'is_premium_only'     => true,
                'has_addons'          => false,
                'has_paid_plans'      => true,
                'is_org_compliant'    => false,
                'menu'                => array(
                    'slug'           => 'woo-gsheet-sync',
                    'first-path'     => 'admin.php?page=woo-gsheet-sync',
                    'support'        => false,
                ),
            ) );
        }

        return $wgs_fs;
    }

    // Init Freemius.
    wgs_fs();
    // Signal that SDK was initiated.
    do_action( 'wgs_fs_loaded' );
}
if (!defined('ABSPATH')) exit;

class Woo_GSheet_Sync {
    private $option_name = 'woo_gsheet_sync_settings';
    private $table_name;
    private $sheets_table_name;
    private $client_id = '';
    private $client_secret = '';
    private $redirect_uri; 


    private $columns = [
        'Order ID',
        'Date',
        'Name',
        'Phone',
        'Address',
        'City',
        'State',
        'Product',
        'Price',
        'Quantity',
        'Variation',
        'Shipping Price',
        'Total',
        'Status',
        'Email'
    ];

    public function __construct() {
        global $wpdb;
        $this->table_name = $wpdb->prefix . 'woo_gsheet_sync';
        $this->sheets_table_name = $wpdb->prefix . 'woo_gsheet_sheets';
        $this->redirect_uri = admin_url('admin.php?page=woo-gsheet-sync');

        // Add action to update order status in Google Sheet
        add_action('woocommerce_order_status_changed', [$this, 'update_order_status_in_google_sheet'], 10, 4);

        // Load credentials from database
        $this->load_credentials();

        add_action('admin_menu', [$this, 'add_plugin_menus']);
        add_action('admin_init', [$this, 'register_settings']);
        add_action('admin_init', [$this, 'handle_oauth_callback']);
        add_action('woocommerce_thankyou', [$this, 'send_order_to_sheet'], 10, 1);
        register_activation_hook(__FILE__, [$this, 'create_plugin_tables']);
        register_deactivation_hook(__FILE__, [$this, 'cleanup_on_deactivation']);

        // Register REST API endpoints
        add_action('rest_api_init', [$this, 'register_rest_routes']);

        // Load composer autoloader
        $this->load_dependencies();
         
        
    }




    /************************************
     * REST API ENDPOINTS
     ************************************/
    public function register_rest_routes() {
        register_rest_route('woo-gsheet-sync/v1', '/sync-order/(?P<id>\d+)', [
            'methods' => 'POST',
            'callback' => [$this, 'api_sync_order'],
            'permission_callback' => [$this, 'api_permissions_check'],
            'args' => [
                'id' => [
                    'validate_callback' => function($param) {
                        return is_numeric($param);
                    }
                ]
            ]
        ]);

        register_rest_route('woo-gsheet-sync/v1', '/sheets', [
            'methods' => 'GET',
            'callback' => [$this, 'api_get_sheets'],
            'permission_callback' => [$this, 'api_permissions_check']
        ]);

        register_rest_route('woo-gsheet-sync/v1', '/sync-status/(?P<id>\d+)', [
            'methods' => 'GET',
            'callback' => [$this, 'api_get_sync_status'],
            'permission_callback' => [$this, 'api_permissions_check']
        ]);
    }

    public function api_permissions_check($request) {
        return current_user_can('manage_options');
    }

    public function api_sync_order($request) {
        $order_id = $request->get_param('id');
        $order = wc_get_order($order_id);
        
        if (!$order) {
            return new WP_Error('not_found', 'Order not found', ['status' => 404]);
        }

        $this->send_order_to_sheet($order_id);
        
        return rest_ensure_response([
            'success' => true,
            'order_id' => $order_id,
            'message' => 'Order sync initiated'
        ]);
    }

    public function api_get_sheets($request) {
        global $wpdb;
        $user_id = get_current_user_id();
        
        $connection = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->table_name} WHERE user_id = %d", 
            $user_id
        ));

        if (!$connection) {
            return new WP_Error('no_connection', 'No Google Sheet connection found', ['status' => 404]);
        }

        $sheets = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$this->sheets_table_name} WHERE connection_id = %d",
            $connection->id
        ));

        return rest_ensure_response([
            'success' => true,
            'sheets' => $sheets
        ]);
    }

    public function api_get_sync_status($request) {
        $order_id = $request->get_param('id');
        $order = wc_get_order($order_id);
        
        if (!$order) {
            return new WP_Error('not_found', 'Order not found', ['status' => 404]);
        }

        $status = $order->get_meta('_gsheet_sync_status');
        
        return rest_ensure_response([
            'success' => true,
            'order_id' => $order_id,
            'synced' => !empty($status),
            'status' => $status ?: 'Not synced'
        ]);
    }

    /************************************
     * ORIGINAL PLUGIN CODE (UNCHANGED)
     ************************************/
    private function load_credentials() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'gsheet_credentials';

        // Check if table exists
        $table_exists = $wpdb->get_var($wpdb->prepare(
            "SHOW TABLES LIKE %s", 
            $wpdb->esc_like($table_name)
        )) === $table_name;

        if ($table_exists) {
            $credentials = $wpdb->get_row("SELECT * FROM $table_name LIMIT 1");
            if ($credentials) {
                $this->client_id = $credentials->client_id;
                $this->client_secret = $credentials->client_secret;
            } else {
                error_log('Woo GSheet Sync: No credentials found in database');
            }
        } else {
            error_log('Woo GSheet Sync: Credentials table does not exist');
        }
    }

    private function load_dependencies() {
        if (!file_exists(__DIR__ . '/vendor/autoload.php')) {
            add_action('admin_notices', [$this, 'show_missing_dependencies_notice']);
            return false;
        }
        require_once __DIR__ . '/vendor/autoload.php';
        return true;
    }

    public function show_missing_dependencies_notice() {
        echo '<div class="notice notice-error"><p>';
        echo 'Google Sheets Sync requires Composer dependencies. Please run <code>composer install</code> in the plugin directory.';
        echo '</p></div>';
    }

    public function create_plugin_tables() {
        global $wpdb;
        $charset_collate = $wpdb->get_charset_collate();

        // Main connection table
        $sql = "CREATE TABLE IF NOT EXISTS {$this->table_name} (
            id INT NOT NULL AUTO_INCREMENT,
            user_id INT NOT NULL,
            spreadsheet_id VARCHAR(255),
            access_token TEXT,
            refresh_token TEXT,
            token_expires INT,
            connected TINYINT(1) DEFAULT 0,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY user_id (user_id)
        ) $charset_collate;";

        // Sheets table to track multiple sheets
        $sql2 = "CREATE TABLE IF NOT EXISTS {$this->sheets_table_name} (
            id INT NOT NULL AUTO_INCREMENT,
            connection_id INT NOT NULL,
            sheet_name VARCHAR(100) NOT NULL,
            sheet_id VARCHAR(255) NOT NULL,
            is_active TINYINT(1) DEFAULT 0,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            FOREIGN KEY (connection_id) REFERENCES {$this->table_name}(id) ON DELETE CASCADE
        ) $charset_collate;";

        // Credentials table
        $sql3 = "CREATE TABLE IF NOT EXISTS {$wpdb->prefix}gsheet_credentials (
            id INT(11) NOT NULL AUTO_INCREMENT,
            client_id TEXT NOT NULL,
            client_secret TEXT NOT NULL,
            PRIMARY KEY (id)
        ) $charset_collate;";

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
        dbDelta($sql2);
        dbDelta($sql3);
    }

    public function cleanup_on_deactivation() {
        delete_transient('woo_gsheet_sync_last_error');
    }

    public function add_plugin_menus() {
        add_menu_page(
            'Woo GSheet Sync',
            'Woo GSheet Sync',
            'manage_options',
            'woo-gsheet-sync',
            [$this, 'settings_page_html'],
            'dashicons-google',
            56
        );

        add_submenu_page(
            'woo-gsheet-sync',
            'Settings',
            'Settings',
            'manage_options',
            'woo-gsheet-sync',
            [$this, 'settings_page_html']
        );

        add_submenu_page(
            'woo-gsheet-sync',
            'Order List',
            'Order List',
            'manage_options',
            'woo-gsheet-orders',
            [$this, 'orders_page_html']
        );
        
        add_submenu_page(
            'woo-gsheet-sync',
            'Credentials',
            'Credentials',
            'manage_options',
            'woo-gsheet-credentials',
            [$this, 'credentials_page_html']
        );
add_submenu_page(
    'woo-gsheet-sync',
    'Service Account details',
    'ServiceAccount',
    'manage_options',
    'woo-gsheet-ServiceAccount',
    [$this, 'service_account_page_html'] // Changed from credentials_page_html
);
    }
     

public function service_account_page_html() {
    $json_file = WP_PLUGIN_DIR . '/WooCommerce Google Sheet Sync/credentials.json';
    
    try {
        // Handle form submission
        if (isset($_POST['save_service_account'])) {
            check_admin_referer('save_gsheet_service_account');
            
            if (empty($_POST['service_account_key'])) {
                throw new Exception('JSON content cannot be empty');
            }

            $json_data = stripslashes($_POST['service_account_key']);
            $decoded = json_decode($json_data);
            
            if (json_last_error() !== JSON_ERROR_NONE || !$decoded) {
                throw new Exception('Invalid JSON: ' . json_last_error_msg());
            }

            // Verify required Google fields
            $required_fields = ['type', 'project_id', 'private_key', 'client_email'];
            foreach ($required_fields as $field) {
                if (!isset($decoded->$field)) {
                    throw new Exception("Missing required field: $field");
                }
            }

            // Attempt to write file
            $result = file_put_contents($json_file, $json_data);
            
            if ($result === false) {
                throw new Exception('Failed to write file. Check directory permissions.');
            }

            // Set secure permissions
            if (!chmod($json_file, 0600)) {
                throw new Exception('Failed to set file permissions');
            }

            echo '<div class="updated"><p>Credentials saved successfully!</p></div>';
        }

        // Read existing file
        $current_content = '';
        if (file_exists($json_file)) {
            if (!is_readable($json_file)) {
                throw new Exception('File exists but is not readable');
            }
            $current_content = file_get_contents($json_file);
        }

    } catch (Exception $e) {
        echo '<div class="error"><p>Error: ' . esc_html($e->getMessage()) . '</p></div>';
        error_log('Woo GSheet Sync Error: ' . $e->getMessage());
    }

    // Display form
    ?>
    <div class="wrap">
        <h1>Google Service Account Configuration</h1>
        
        <div class="notice notice-info">
            <p>Edit your Google Service Account credentials directly. The file will be saved to:</p>
            <code><?php echo esc_html($json_file); ?></code>
            <?php if (!file_exists($json_file)) : ?>
                <p class="notice-warning" style="color:#d63638;">File doesn't exist yet. It will be created on first save.</p>
            <?php endif; ?>
        </div>

        <form method="post">
            <?php wp_nonce_field('save_gsheet_service_account'); ?>
            <table class="form-table">
                <tr>
                    <th scope="row">Service Account JSON</th>
                    <td>
                        <textarea name="service_account_key" id="service_account_key" 
                                rows="20" cols="100" class="large-text code" required
                                placeholder='{"type": "service_account", ...}'><?php 
                                echo esc_textarea($current_content); ?></textarea>
                        <p class="description">
                            Paste the complete JSON key obtained from Google Cloud Console.
                        </p>
                    </td>
                </tr>
            </table>
            
            <p class="submit">
                <input type="submit" name="save_service_account" class="button button-primary" value="Save Credentials">
                <?php if (file_exists($json_file)) : ?>
                    <a href="<?php echo esc_url(plugin_dir_url(__FILE__) . 'credentials.json?t=' . time()); ?>" 
                       class="button" download>Download Current</a>
                <?php endif; ?>
            </p>
        </form>
    </div>
    <?php
}


    public function credentials_page_html() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'gsheet_credentials';

        // Handle form submission
        if (isset($_POST['save_credentials'])) {
            check_admin_referer('save_gsheet_credentials');
            
            $client_id = sanitize_text_field($_POST['client_id']);
            $client_secret = sanitize_text_field($_POST['client_secret']);

            $existing = $wpdb->get_row("SELECT * FROM $table_name LIMIT 1");

            if ($existing) {
                $wpdb->update(
                    $table_name,
                    ['client_id' => $client_id, 'client_secret' => $client_secret],
                    ['id' => $existing->id]
                );
                echo '<div class="updated"><p>Credentials updated successfully.</p></div>';
            } else {
                $wpdb->insert(
                    $table_name,
                    ['client_id' => $client_id, 'client_secret' => $client_secret]
                );
                echo '<div class="updated"><p>Credentials saved successfully.</p></div>';
            }
        }

        // Fetch saved values (if any)
        $credentials = $wpdb->get_row("SELECT * FROM $table_name LIMIT 1");

        $client_id_value = isset($credentials->client_id) ? esc_attr($credentials->client_id) : '';
        $client_secret_value = isset($credentials->client_secret) ? esc_attr($credentials->client_secret) : '';

        // Output form
        ?>
        <div class="wrap">
            <h1>Google API Credentials</h1>
            <form method="post">
                <?php wp_nonce_field('save_gsheet_credentials'); ?>
                <table class="form-table">
                    <tr>
                        <th scope="row"><label for="client_id">Client ID</label></th>
                        <td><input name="client_id" type="text" id="client_id" value="<?php echo $client_id_value; ?>" class="regular-text" required></td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="client_secret">Client Secret</label></th>
                        <td><input name="client_secret" type="text" id="client_secret" value="<?php echo $client_secret_value; ?>" class="regular-text" required></td>
                    </tr>
                </table>
                <p class="submit">
                    <input type="submit" name="save_credentials" id="submit" class="button button-primary" value="Save Credentials">
                </p>
            </form>
        </div>
        <?php
    }

    public function register_settings() {
        register_setting('woo_gsheet_sync_group', $this->option_name);
    }

    public function handle_oauth_callback() {
        if (!isset($_GET['page']) || $_GET['page'] !== 'woo-gsheet-sync') {
            return;
        }

        if (isset($_GET['code'])) {
            $this->process_oauth_callback();
        }

        if (isset($_GET['disconnect'])) {
            $this->disconnect_account();
        }

        if (isset($_GET['delete_sheet']) && isset($_GET['sheet_id'])) {
            $this->delete_sheet(intval($_GET['sheet_id']));
        }

        if (isset($_POST['save_sheet_settings'])) {
            $this->save_sheet_settings();
        }

        if (isset($_POST['set_active_sheet'])) {
            $this->set_active_sheet(intval($_POST['active_sheet']));
        }
    }

private function process_oauth_callback() {
    if (!$this->load_dependencies()) {
        $this->add_error_notice('Google API client not loaded. Please install dependencies.');
        return;
    }

    if (empty($this->client_id) || empty($this->client_secret)) {
        $this->add_error_notice('Google API credentials not configured. Please set them in the Credentials page.');
        return;
    }

    try {
        $client = new Google_Client();
        $client->setClientId($this->client_id);
        $client->setClientSecret($this->client_secret);
        $client->setRedirectUri($this->redirect_uri);
        $client->addScope([
            Google_Service_Sheets::SPREADSHEETS,
            Google_Service_Drive::DRIVE_FILE
        ]);
        $client->setAccessType('offline');
        $client->setPrompt('consent');
        $client->setIncludeGrantedScopes(true);

        // Debug: Log the auth URL for verification
        error_log('Google Auth URL: ' . $client->createAuthUrl());

        $token = $client->fetchAccessTokenWithAuthCode($_GET['code']);
        
        if (isset($token['error'])) {
            throw new Exception($token['error_description']);
        }

        // Verify we got all required token data
        $required_token_fields = ['access_token', 'expires_in', 'refresh_token'];
        foreach ($required_token_fields as $field) {
            if (!isset($token[$field])) {
                throw new Exception("Missing required token field: $field");
            }
        }

        // Save token data
        global $wpdb;
        $user_id = get_current_user_id();
        $data = [
            'user_id' => $user_id,
            'access_token' => json_encode($token),
            'refresh_token' => $token['refresh_token'],
            'token_expires' => time() + $token['expires_in'],
            'connected' => 1
        ];

        $existing = $wpdb->get_row($wpdb->prepare(
            "SELECT id FROM {$this->table_name} WHERE user_id = %d", 
            $user_id
        ));

        if ($existing) {
            $wpdb->update(
                $this->table_name,
                $data,
                ['user_id' => $user_id],
                ['%d', '%s', '%s', '%d', '%d'],
                ['%d']
            );
        } else {
            $wpdb->insert($this->table_name, $data);
        }

        $this->add_success_notice('Google account connected successfully!');
        
    } catch (Exception $e) {
        error_log('Google OAuth Error: ' . $e->getMessage());
        $this->add_error_notice('Error connecting Google account: ' . $e->getMessage());
        
        // Additional debug info:
        error_log('Client ID: ' . $this->client_id);
        error_log('Redirect URI: ' . $this->redirect_uri);
        if (isset($_GET['code'])) {
            error_log('Auth code received');
        } else {
            error_log('No auth code in request');
        }
    }
}

    private function disconnect_account() {
        global $wpdb;
        $user_id = get_current_user_id();
        
        $wpdb->delete(
            $this->table_name,
            ['user_id' => $user_id],
            ['%d']
        );
        
        $this->add_success_notice('Google account disconnected successfully.');
    }

    private function delete_sheet($sheet_id) {
        global $wpdb;
        $user_id = get_current_user_id();
        
        // Verify the sheet belongs to the current user's connection
        $sheet = $wpdb->get_row($wpdb->prepare(
            "SELECT s.* FROM {$this->sheets_table_name} s 
            JOIN {$this->table_name} c ON s.connection_id = c.id 
            WHERE s.id = %d AND c.user_id = %d",
            $sheet_id, $user_id
        ));

        if (!$sheet) {
            $this->add_error_notice('Sheet not found or you do not have permission to delete it.');
            return;
        }

        $wpdb->delete(
            $this->sheets_table_name,
            ['id' => $sheet_id],
            ['%d']
        );

        $this->add_success_notice('Sheet deleted successfully.');
    }

    private function set_active_sheet($sheet_id) {
        global $wpdb;
        $user_id = get_current_user_id();
        
        // Verify the sheet belongs to the current user's connection
        $sheet = $wpdb->get_row($wpdb->prepare(
            "SELECT s.* FROM {$this->sheets_table_name} s 
            JOIN {$this->table_name} c ON s.connection_id = c.id 
            WHERE s.id = %d AND c.user_id = %d",
            $sheet_id, $user_id
        ));

        if (!$sheet) {
            $this->add_error_notice('Sheet not found or you do not have permission to activate it.');
            return;
        }

        // First set all sheets to inactive
        $wpdb->update(
            $this->sheets_table_name,
            ['is_active' => 0],
            ['connection_id' => $sheet->connection_id],
            ['%d'],
            ['%d']
        );

        // Then activate the selected sheet
        $wpdb->update(
            $this->sheets_table_name,
            ['is_active' => 1],
            ['id' => $sheet_id],
            ['%d'],
            ['%d']
        );

        $this->add_success_notice('Active sheet updated successfully.');
    }

    private function add_error_notice($message) {
        add_settings_error(
            'woo_gsheet_sync_messages',
            'woo_gsheet_sync_message',
            __($message, 'woo-gsheet-sync'),
            'error'
        );
    }

    private function add_success_notice($message) {
        add_settings_error(
            'woo_gsheet_sync_messages',
            'woo_gsheet_sync_message',
            __($message, 'woo-gsheet-sync'),
            'success'
        );
    }

    public function settings_page_html() {
        global $wpdb;
        $user_id = get_current_user_id();
        $connection = $wpdb->get_row(
            $wpdb->prepare("SELECT * FROM {$this->table_name} WHERE user_id = %d", $user_id)
        );

        settings_errors('woo_gsheet_sync_messages');
        ?>
        <div class="wrap">
            <h1>WooCommerce Google Sheet Sync - Settings</h1>
            
            <?php if ($connection && $connection->connected): ?>
<div class="card-container">
    <div class="card-column">
        <div class="card">
            <h2>Connected Google Account</h2>
            <p>Your Google account is successfully connected.</p>
            <form method="post" action="<?php echo esc_url(admin_url('admin.php?page=woo-gsheet-sync&disconnect=1')); ?>">
                <?php submit_button('Disconnect Account', 'delete', 'disconnect', false); ?>
            </form>
        </div>
    </div>
    
    <div class="card-column">
        <div class="card">
            <h2>Spreadsheet Settings</h2>
            <form method="post">
                <table class="form-table vertical-labels">
                    <tr valign="top">
                        <td colspan="2">
                            <label for="new_sheet_name">Sheet Name</label>
                            <input type="text" name="new_sheet_name" id="new_sheet_name" class="regular-text" required />
                            <p class="description">Enter a name for the new sheet</p>
                        </td>
                    </tr>
                    <tr valign="top" style="display:none;">
                        <td colspan="2">
                            <label for="spreadsheet_id">Spreadsheet ID</label>
                            <input type="text" name="spreadsheet_id" id="spreadsheet_id" class="regular-text" 
                                value="<?php //echo esc_attr($connection->spreadsheet_id ?? ''); ?>" />
                            <p class="description">Leave blank to create a new spreadsheet</p>
                        </td>
                    </tr>
                    <tr valign="top" style="display:none;">
                        <td colspan="2">
                            <label for="folder_id">Folder ID (optional)</label>
                            <input type="text" name="folder_id" id="folder_id" class="regular-text" />
                            <p class="description">Leave blank to save in your Google Drive root</p>
                        </td>
                    </tr>
                </table>
                <?php submit_button('Create New Sheet', 'primary', 'save_sheet_settings'); ?>
            </form>
        </div>
    </div>
    <div class="card-column">
<div class="card" style="overflow-x: auto;">
    <h2>Your Sheets</h2>
    <?php
    $sheets = $wpdb->get_results($wpdb->prepare(
        "SELECT * FROM {$this->sheets_table_name} 
        WHERE connection_id = %d 
        ORDER BY is_active DESC, created_at DESC",
        $connection->id
    ));

    if (!empty($sheets)): ?>
        <table class="wp-list-table widefat fixed striped" style="width: 100%; table-layout: auto;">
            <thead>
                <tr>
                    <th style="width: 25%;">Sheet Name</th>
                    <th style="width: 35%;">Sheet ID</th>
                    <th style="width: 20%;">Status</th>
                    <th style="width: 20%;">Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($sheets as $sheet): ?>
                    <tr>
                        <td style="width: 25%;"><?php echo esc_html($sheet->sheet_name); ?></td>
                        <td style="width: 35%; word-break: break-all;"><?php echo esc_html($sheet->sheet_id); ?></td>
                        <td style="width: 20%;">
                            <?php if ($sheet->is_active): ?>
                                <span class="dashicons dashicons-yes" style="color: #46b450;"></span> Active
                            <?php else: ?>
                                <form method="post" style="display: inline;">
                                    <input type="hidden" name="active_sheet" value="<?php echo esc_attr($sheet->id); ?>">
                                    <?php submit_button('Set Active', 'small', 'set_active_sheet', false); ?>
                                </form>
                            <?php endif; ?>
                        </td>
                        <td style="width: 20%;">
                            <a href="<?php echo esc_url(admin_url('admin.php?page=woo-gsheet-sync&delete_sheet=1&sheet_id=' . $sheet->id)); ?>" 
                               class="button button-small" 
                               onclick="return confirm('Are you sure you want to delete this sheet?');">
                                Delete
                            </a>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    <?php else: ?>
        <p>No sheets created yet. Create your first sheet above.</p>
    <?php endif; ?>
</div>
    </div>
</div>


<style>

    .card {
        max-width: 100%;
    }
    
    @media screen and (max-width: 782px) {
        .wp-list-table th, 
        .wp-list-table td {
            display: table-cell;
            padding: 8px 10px;
        }
    }
    .card-container {
        display: flex;
        flex-wrap: wrap;
        gap: 20px;
        margin-bottom: 20px;
    }
    
    .card-column {
        flex: 3;
    }
    
    .card {
        height: 100%;
        box-sizing: border-box;
    }
    
    /* Vertical label styling */
    .vertical-labels td {
        display: block;
        padding: 5px 0;
    }
    
    .vertical-labels label {
        display: block;
        margin-bottom: 5px;
        font-weight: 600;
    }
    
    .vertical-labels input.regular-text {
        width: 100%;
        max-width: 100%;
    }
    
    @media (max-width: 782px) {
        .card-column {
            flex: 100%;
        }
    }
</style>
            <?php else: ?>
                <div class="card">
                    <h2>Connect Google Account</h2>
                    <p>Click the button below to connect your Google account:</p>
                    <?php
                    if (empty($this->client_id) || empty($this->client_secret)) {
                        echo '<div class="notice notice-error"><p>Please configure your Google API credentials first on the Credentials page.</p></div>';
                    } else {
                        $client = new Google_Client();
                        $client->setClientId($this->client_id);
                        $client->setClientSecret($this->client_secret);
                        $client->setRedirectUri($this->redirect_uri);
                        $client->addScope([
                            Google_Service_Sheets::SPREADSHEETS,
                            Google_Service_Drive::DRIVE_FILE
                        ]);
                        $client->setAccessType('offline');
                        $client->setPrompt('consent');
                        
                        $auth_url = $client->createAuthUrl();
                        ?>
                        <a href="<?php echo esc_url($auth_url); ?>" class="button button-primary">Connect Google Account</a>
                        <p class="description">You'll be redirected to Google to authorize access</p>
                    <?php } ?>
                </div>
            <?php endif; ?>
        </div>
        <?php
    }

    private function save_sheet_settings() {
        global $wpdb;
        $user_id = get_current_user_id();
        $connection = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->table_name} WHERE user_id = %d", 
            $user_id
        ));

        if (!$connection) {
            $this->add_error_notice('No active connection found.');
            return;
        }

        $sheet_name = sanitize_text_field($_POST['new_sheet_name'] ?? '');

        if (empty($sheet_name)) {
            $this->add_error_notice('Sheet name is required.');
            return;
        }

        $result = $this->create_google_sheet($sheet_name);

        if ($result) {
            $this->add_success_notice('New sheet created successfully!');
        }
    }

    private function create_google_sheet($sheet_name, $folder_id = '') {
        global $wpdb;
        $user_id = get_current_user_id();
        $connection = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->table_name} WHERE user_id = %d", 
            $user_id
        ));

        if (!$connection || !$connection->connected) {
            $this->add_error_notice('Please connect your Google account first.');
            return false;
        }

        try {
            $client = $this->get_google_client();
            if (!$client) {
                throw new Exception('Failed to initialize Google client');
            }

            $token = json_decode($connection->access_token, true);
            if (json_last_error() !== JSON_ERROR_NONE) {
                throw new Exception('Invalid token data');
            }

            $client->setAccessToken($token);

            if ($client->isAccessTokenExpired()) {
                if (empty($connection->refresh_token)) {
                    throw new Exception('No refresh token available');
                }

                $new_token = $client->fetchAccessTokenWithRefreshToken($connection->refresh_token);
                $client->setAccessToken($new_token);
                
                $wpdb->update(
                    $this->table_name,
                    [
                        'access_token' => json_encode($new_token),
                        'token_expires' => time() + $new_token['expires_in']
                    ],
                    ['user_id' => $user_id],
                    ['%s', '%d'],
                    ['%d']
                );
            }

            $sheetsService = new Google_Service_Sheets($client);
            $driveService = new Google_Service_Drive($client);

            // Create new spreadsheet
            $spreadsheet = new Google_Service_Sheets_Spreadsheet([
                'properties' => [
                    'title' => $sheet_name
                ]
            ]);

            $spreadsheet = $sheetsService->spreadsheets->create($spreadsheet);
            $spreadsheet_id = $spreadsheet->spreadsheetId;

            // Move to folder if specified
            if (!empty($folder_id)) {
                $file = new Google_Service_Drive_DriveFile();
                $file->setParents([$folder_id]);
                $driveService->files->update($spreadsheet_id, $file);
            }

            // Rename the first sheet
            $sheet_properties = new Google_Service_Sheets_SheetProperties();
            $sheet_properties->setSheetId(0);
            $sheet_properties->setTitle($sheet_name);

            $updateRequest = new Google_Service_Sheets_UpdateSheetPropertiesRequest();
            $updateRequest->setProperties($sheet_properties);
            $updateRequest->setFields('title');

            $batchUpdateRequest = new Google_Service_Sheets_BatchUpdateSpreadsheetRequest([
                'requests' => [
                    new Google_Service_Sheets_Request([
                        'updateSheetProperties' => $updateRequest
                    ])
                ]
            ]);
            
            $sheetsService->spreadsheets->batchUpdate($spreadsheet_id, $batchUpdateRequest);

            // Add headers using the defined columns
            $range = $sheet_name . '!A1:' . chr(64 + count($this->columns)) . '1';
            $header_values = [$this->columns];
            
            $body = new Google_Service_Sheets_ValueRange([
                'values' => $header_values
            ]);
            
            $sheetsService->spreadsheets_values->update(
                $spreadsheet_id,
                $range,
                $body,
                ['valueInputOption' => 'RAW']
            );

            // Check if this is the first sheet being created
            $is_first_sheet = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM {$this->sheets_table_name} WHERE connection_id = %d",
                $connection->id
            )) == 0;

            // Add to sheets table
            $wpdb->insert(
                $this->sheets_table_name,
                [
                    'connection_id' => $connection->id,
                    'sheet_name' => $sheet_name,
                    'sheet_id' => $spreadsheet_id,
                    'is_active' => $is_first_sheet ? 1 : 0
                ],
                ['%d', '%s', '%s', '%d']
            );

            if ($is_first_sheet) {
                $this->add_success_notice(sprintf(
                    'First spreadsheet created and set as active! <a href="%s" target="_blank">Open Sheet</a>',
                    esc_url('https://docs.google.com/spreadsheets/d/' . $spreadsheet_id)
                ));
            } else {
                $this->add_success_notice(sprintf(
                    'New spreadsheet created! <a href="%s" target="_blank">Open Sheet</a>',
                    esc_url('https://docs.google.com/spreadsheets/d/' . $spreadsheet_id)
                ));
            }

            return true;
        } catch (Exception $e) {
            error_log('Google Sheets creation error: ' . $e->getMessage());
            $this->add_error_notice('Error creating spreadsheet: ' . $e->getMessage());
            return false;
        }
    }

    private function get_google_client() {
        if (empty($this->client_id) || empty($this->client_secret)) {
            $this->add_error_notice('Google API credentials not configured. Please set them in the Credentials page.');
            return false;
        }

        if (!class_exists('Google_Client')) {
            $this->add_error_notice('Google API client library not loaded. Please install dependencies.');
            return false;
        }

        $client = new Google_Client();
        $client->setClientId($this->client_id);
        $client->setClientSecret($this->client_secret);
        $client->addScope([
            Google_Service_Sheets::SPREADSHEETS,
            Google_Service_Drive::DRIVE_FILE
        ]);
        $client->setAccessType('offline');
        
        return $client;
    }

    public function orders_page_html() {
        global $wpdb;
        $user_id = get_current_user_id();
        $connection = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->table_name} WHERE user_id = %d", 
            $user_id
        ));

        echo '<div class="wrap">';
        echo '<h1>WooCommerce Google Sheet Sync - Order List</h1>';
        
        if ($connection && $connection->connected) {
            // Get the active sheet
            $active_sheet = $wpdb->get_row($wpdb->prepare(
                "SELECT * FROM {$this->sheets_table_name} 
                WHERE connection_id = %d AND is_active = 1",
                $connection->id
            ));

            if ($active_sheet) {
                echo '<p>Orders are being synced to: <strong>' . esc_html($active_sheet->sheet_name) . '</strong> in ';
                echo '<a href="' . esc_url('https://docs.google.com/spreadsheets/d/' . $active_sheet->sheet_id) . '" target="_blank">';
                echo esc_html($active_sheet->sheet_id);
                echo '</a></p>';
                
                try {
                    // Get data from the active sheet
                    $client = $this->get_google_client();
                    $client->setAccessToken(json_decode($connection->access_token, true));
                    
                    if ($client->isAccessTokenExpired()) {
                        $client->fetchAccessTokenWithRefreshToken($connection->refresh_token);
                    }
                    
                    $service = new Google_Service_Sheets($client);
                    $range = $active_sheet->sheet_name . '!A2:O';
                    
                    $response = $service->spreadsheets_values->get($active_sheet->sheet_id, $range);
                    $values = $response->getValues();
                    
                    if (empty($values)) {
                        echo '<div class="notice notice-info"><p>No orders found in the active sheet.</p></div>';
                    } else {
                        echo '<h3>Orders Synced to Active Sheet</h3>';
                        echo '<div style="overflow-x: auto;">';
                        echo '<table class="wp-list-table widefat fixed striped">';
                        echo '<thead><tr>';
                        
                        foreach ($this->columns as $column) {
                            echo '<th>' . esc_html($column) . '</th>';
                        }
                        
                        echo '</tr></thead>';
                        echo '<tbody>';
                        
                        foreach ($values as $row) {
                            echo '<tr>';
                            for ($i = 0; $i < 15; $i++) {
                                echo '<td>' . esc_html($row[$i] ?? '') . '</td>';
                            }
                            echo '</tr>';
                        }
                        
                        echo '</tbody></table>';
                        echo '</div>';
                    }
                } catch (Exception $e) {
                    echo '<div class="notice notice-error"><p>Error fetching data from Google Sheet: ' . esc_html($e->getMessage()) . '</p></div>';
                    error_log('Google Sheets fetch error: ' . $e->getMessage());
                }
            } else {
                echo '<div class="notice notice-warning"><p>No active sheet selected. Please select one in the Settings page.</p></div>';
            }
            
            // Show recent WooCommerce orders for reference
            $orders = wc_get_orders([
                'limit' => 5,
                'orderby' => 'date',
                'order' => 'DESC'
            ]);
            
            if (!empty($orders)) {
                echo '<h3>Recent WooCommerce Orders (Reference)</h3>';
                echo '<table class="wp-list-table widefat fixed striped">';
                echo '<thead><tr>
                    <th>Order #</th>
                    <th>Customer</th>
                    <th>Total</th>
                    <th>Date</th>
                    <th>Sync Status</th>
                </tr></thead>';
                echo '<tbody>';
                
                foreach ($orders as $order) {
                    $sync_status = $order->get_meta('_gsheet_sync_status');
                    echo '<tr>';
                    echo '<td>' . esc_html($order->get_id()) . '</td>';
                    echo '<td>' . esc_html($order->get_billing_first_name() . ' ' . $order->get_billing_last_name()) . '</td>';
                    echo '<td>' . wc_price($order->get_total()) . '</td>';
                    echo '<td>' . esc_html(wc_format_datetime($order->get_date_created())) . '</td>';
                    echo '<td>' . ($sync_status ? esc_html($sync_status) : 'Not synced') . '</td>';
                    echo '</tr>';
                }
                
                echo '</tbody></table>';
            }
        } else {
            echo '<div class="notice notice-warning"><p>No Google Sheet connection established yet.</p></div>';
        }
        
        echo '</div>';
    }

    public function send_order_to_sheet($order_id) {
        global $wpdb;
        $user_id = get_current_user_id();
        $connection = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->table_name} WHERE user_id = %d AND connected = 1", 
            $user_id
        ));

        if (!$connection) {
            error_log('No active Google Sheet connection for order ' . $order_id);
            return;
        }

        // Get the active sheet for this connection
        $active_sheet = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->sheets_table_name} 
            WHERE connection_id = %d AND is_active = 1",
            $connection->id
        ));

        if (!$active_sheet) {
            error_log('No active sheet selected for order ' . $order_id);
            return;
        }

        $order = wc_get_order($order_id);
        if (!$order) {
            return;
        }

        // Prepare order data according to the defined columns
        $data = [[
            $order->get_id(),
            $order->get_date_created()->date('Y-m-d H:i:s'),
            $order->get_billing_first_name() . ' ' . $order->get_billing_last_name(),
            $order->get_billing_phone(),
            $order->get_billing_address_1(),
            $order->get_billing_city(),
            $order->get_billing_state(),
            implode(', ', array_map(function($item) {
                return $item->get_name();
            }, $order->get_items())),
            $order->get_subtotal(),
            array_sum(array_map(function($item) {
                return $item->get_quantity();
              }, $order->get_items())),
       
       
        implode(', ', array_map(function($item) {
        if ($item->get_variation_id()) {
            $variation = wc_get_product($item->get_variation_id());
            if ($variation) {
                $attributes = $variation->get_attributes();
                return implode(', ', array_values($attributes));
            }
        }
        return '';
    }, $order->get_items())),
            $order->get_shipping_total(),
            $order->get_total(),
            $order->get_status(),
            $order->get_billing_email()
        ]];

        $result = $this->append_to_sheet(
            $active_sheet->sheet_id, 
            $active_sheet->sheet_name, 
            $data
        );
        
        if ($result === false) {
            $order->add_order_note('Failed to sync order to Google Sheet');
            $order->update_meta_data('_gsheet_sync_status', 'Failed');
        } else {
            $order->add_order_note('Order synced to Google Sheet successfully');
            $order->update_meta_data('_gsheet_sync_status', 'Synced');
        }
        $order->save();
    }

    public function update_order_status_in_google_sheet($order_id, $old_status, $new_status, $order) {
        if (!is_admin()) return;

        if (!$order instanceof WC_Order) return;

        global $wpdb;

        $user_id = get_current_user_id();

        // Get connection
        $connection = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->table_name} WHERE user_id = %d AND connected = 1",
            $user_id
        ));

        if (!$connection) {
            $order->add_order_note("Google Sheet Sync: No active connection.");
            error_log('Woo GSheet Sync: No active connection found.');
            return;
        }

        // Get active sheet
        $active_sheet = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->sheets_table_name} WHERE connection_id = %d AND is_active = 1",
            $connection->id
        ));

        if (!$active_sheet) {
            $order->add_order_note("Google Sheet Sync: No active sheet found.");
            error_log('Woo GSheet Sync: No active sheet found.');
            return;
        }

        try {
            $client = new Google_Client();
            $client->setClientId($this->client_id);
            $client->setClientSecret($this->client_secret);
            $client->setAccessToken(json_decode($connection->access_token, true));

            if ($client->isAccessTokenExpired()) {
                if (empty($connection->refresh_token)) {
                    throw new Exception("No refresh token available.");
                }

                $new_token = $client->fetchAccessTokenWithRefreshToken($connection->refresh_token);
                $client->setAccessToken($new_token);

                // Save refreshed token
                $wpdb->update(
                    $this->table_name,
                    [
                        'access_token' => json_encode($new_token),
                        'token_expires' => time() + $new_token['expires_in']
                    ],
                    ['user_id' => $user_id],
                    ['%s', '%d'],
                    ['%d']
                );
            }

            $service = new Google_Service_Sheets($client);

            // Read Order IDs from Column A
            $range = $active_sheet->sheet_name . '!A:A';
            $response = $service->spreadsheets_values->get($active_sheet->sheet_id, $range);
            $values = $response->getValues();

            if (empty($values)) {
                throw new Exception("Google Sheet is empty.");
            }

            // Find the row number for this order ID
            $row_number = null;
            foreach ($values as $index => $row) {
                if (isset($row[0]) && $row[0] == $order_id) {
                    $row_number = $index + 1; // Sheets are 1-indexed
                    break;
                }
            }

            if (!$row_number) {
                throw new Exception("Order ID {$order_id} not found in the Google Sheet.");
            }

            // Update the status in column N (13th column)
            $status_range = $active_sheet->sheet_name . '!N' . $row_number;
            $body = new Google_Service_Sheets_ValueRange([
                'values' => [[$new_status]]
            ]);

            $result = $service->spreadsheets_values->update(
                $active_sheet->sheet_id,
                $status_range,
                $body,
                ['valueInputOption' => 'RAW']
            );

            if ($result->getUpdatedCells() > 0) {
                $order->add_order_note("Google Sheet Sync: Status updated to '{$new_status}' in Google Sheet.");
                error_log("Woo GSheet Sync: Updated order status for #{$order_id} to {$new_status}.");
            } else {
                throw new Exception("Status update failed for Order ID {$order_id}.");
            }

        } catch (Exception $e) {
            error_log('Woo GSheet Sync Error: ' . $e->getMessage());
            $order->add_order_note("Google Sheet Sync failed: " . $e->getMessage());
        }
    }

    private function append_to_sheet($spreadsheet_id, $sheet_name, $values) {
        global $wpdb;
        $user_id = get_current_user_id();
        $connection = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->table_name} WHERE user_id = %d", 
            $user_id
        ));

        if (!$connection || !$connection->connected || empty($connection->access_token)) {
            error_log('No valid Google connection found for user ' . $user_id);
            return false;
        }

        try {
            $client = $this->get_google_client();
            if (!$client) {
                throw new Exception('Google client not initialized');
            }

            $token = json_decode($connection->access_token, true);
            if (json_last_error() !== JSON_ERROR_NONE) {
                throw new Exception('Invalid token data');
            }

            $client->setAccessToken($token);

            if ($client->isAccessTokenExpired()) {
                if (empty($connection->refresh_token)) {
                    throw new Exception('No refresh token available');
                }

                $new_token = $client->fetchAccessTokenWithRefreshToken($connection->refresh_token);
                $client->setAccessToken($new_token);
                
                $wpdb->update(
                    $this->table_name,
                    [
                        'access_token' => json_encode($new_token),
                        'token_expires' => time() + $new_token['expires_in']
                    ],
                    ['user_id' => $user_id],
                    ['%s', '%d'],
                    ['%d']
                );
            }

            $service = new Google_Service_Sheets($client);
            $range = $sheet_name . '!A:' . chr(64 + count($this->columns));

            $body = new Google_Service_Sheets_ValueRange([
                'values' => $values
            ]);

            $result = $service->spreadsheets_values->append(
                $spreadsheet_id,
                $range,
                $body,
                ['valueInputOption' => 'RAW']
            );
            
            return $result->getUpdates()->getUpdatedCells();
        } catch (Exception $e) {
            error_log('Google Sheets append error: ' . $e->getMessage());
            set_transient('woo_gsheet_sync_last_error', $e->getMessage(), 60);
            return false;
        }
    }
}

new Woo_GSheet_Sync();